import unittest
from math import sqrt
from random import randrange

import numpy as np

from linalg.Vector import Vector


class TestVector(unittest.TestCase):
    def test_addition_with_scalar_raises(self):
        V = Vector(1, 2, 3)

        self.assertRaises(TypeError, lambda: V + 1)
        self.assertRaises(TypeError, lambda: 1 + V)

    def test_equality(self):
        V = Vector(1, 2, 3)
        ZeroVector = Vector()
        self.assertTrue((V == [1, 2, 3]))
        self.assertTrue((V == np.array([1, 2, 3])))
        self.assertTrue((V == Vector(1, 2, 3)))
        self.assertTrue((V == V))

        self.assertFalse((V == [2, 3]))
        self.assertFalse((V == np.array([1, 2])))
        self.assertFalse((V == Vector(1, 3)))
        self.assertFalse(V == [])
        self.assertFalse(V == ZeroVector)
        self.assertFalse(ZeroVector == V)
        self.assertFalse(ZeroVector == [1])

    def test_vector_addition(self):
        V = Vector(1, 2, 3)

        self.assertTrue((V + V == 2 * V))
        self.assertTrue((V + [2, 2, 3] == [3, 4, 6]))
        self.assertTrue((V + [2, 2, 3] == Vector(3, 4, 6)))

    def test_vector_subtraction(self):
        V = Vector(1, 2, 3)

        self.assertTrue(V - V == Vector(0, 0, 0))
        self.assertTrue(V - [1, 1, 1] == Vector(0, 1, 2))
        self.assertTrue([1, 1, 1] - V == Vector(0, -1, -2))

    def test_multiplication_with_scalar(self):
        V = Vector(1, 2, 3)

        self.assertTrue(V * 3 == Vector(3, 6, 9))
        self.assertTrue(V * (-1) == Vector(-1, -2, -3))

    def test_multiplication_commutativity(self):
        V = Vector(1, 2, 3)

        self.assertTrue((V * 3 == 3 * V))

    def test_transform_vector_in_homogeneous_coordinates(self):
        V = Vector(1, 2, 3)
        ZeroVector = Vector()
        self.assertTrue(V.homog() == [1, 2, 3, 1])
        self.assertTrue(ZeroVector.homog() == ZeroVector)
        self.assertTrue(ZeroVector.homog() == [])

    def test_remove_homogeneous_coordinate(self):
        V = Vector(1, 2, 3)
        HV = Vector(1, 2, 3, 1)
        HV2 = Vector(1, 2, 3, 10)
        self.assertTrue(V.homog().deHomog() == V)
        self.assertTrue(HV.deHomog() == V)
        self.assertTrue(HV2.deHomog() == [0.1, 0.2, 0.3])

    def test_vector_norm_calculation(self):
        n1, n2, n3 = randrange(10), randrange(10), randrange(10)
        self.assertTrue(Vector(0, 0).norm() == 0)
        self.assertTrue(Vector(-1, 0).norm() == 1)
        self.assertTrue(Vector(1, 0).norm() == 1)
        self.assertTrue(Vector(0, 1).norm() == 1)
        self.assertTrue(Vector(n1, n2).norm() == sqrt(n1 ** 2 + n2 ** 2))
        self.assertTrue(Vector(n1, n2, n3).norm() == sqrt(n1 ** 2 + n2 ** 2 + n3 ** 2))

    def test_linear_dependence(self):
        self.assertTrue(Vector(1, 2).linearlyDependent([2, 4]))
        self.assertTrue(Vector(1, 2).linearlyDependent(Vector(2, 4)))
        self.assertFalse(Vector(1, 2).linearlyIndependent([2, 4]))
        self.assertFalse(Vector(1, 2).linearlyIndependent(Vector(2, 4)))
        self.assertTrue(Vector(1, 2).linearlyDependent([0, 0]))
        self.assertTrue(Vector(1, 2, 0, 1).linearlyDependent([0, 0, 0, 0]))
        self.assertTrue(Vector(1, 0, 0, 0).linearlyIndependent([0, 0, 0, 1]))
        self.assertTrue(Vector(0, 0, 1, 0).linearlyIndependent([1, 0, 0, 0]))
        self.assertFalse(Vector(1, 2, 3).linearlyDependent(Vector(4, 8, 10)))
