from __future__ import annotations

import typing

import numpy as np

from linalg.Matrix import Matrix
from linalg.Vector import Vector
from linalg.linmap.maps import identity


def _getBin(n: int, k: int) -> str:
    return str(bin(n)[2:]).zfill(k)


def _z2(v: Vector) -> Vector:
    w = []
    for element in v:
        w.append(int(element % 2))
    return Vector(*w)


class HammingCode:
    def __init__(self, k: int):
        self._k = k
        self._n = 2 ** k - k - 1
        self._N = self._n + self._k
        self._G = self.get_generator_matrix()
        self._H = self.get_check_matrix()

    def get_generator_matrix(self) -> Matrix:
        M = identity(self._n)
        tmp = self.get_check_matrix()
        for i in range(0, self._k):
            tmp = tmp.dropCol(2 ** i - 1 - i)

        for i in range(0, self._k):
            M = M.insertRow(2 ** i - 1, tmp.getRow(self._k - i - 1))

        return M

    def get_check_matrix(self) -> Matrix:
        arr = []
        for i in range(1, 2 ** self._k):
            arr.append([int(b) for b in _getBin(i, self._k)])

        return Matrix(*arr).transpose()

    def encode(self, word: ARRAY_LIKE) -> Vector:
        return _z2(self._G * word)

    def decode(self, codeword: ARRAY_LIKE) -> Vector:
        V = codeword if self.check(codeword) else self._fix(codeword)
        arr = []
        skip = [2 ** i - 1 for i in range(self._k)]
        for i in range(self._N):
            if i not in skip:
                arr.append(int(V[i]))
        return Vector(*arr)

    def check(self, codeword: ARRAY_LIKE) -> bool:
        return (_z2(self.get_check_matrix() * codeword)).isZeroVector()

    def _fix(self, codeword: ARRAY_LIKE) -> Vector:
        v = _z2(self._H * codeword)
        V = [*codeword]
        position = int(''.join([str(b) for b in v]), 2) - 1
        V[position] = (codeword[position] + 1) % 2
        return Vector(*V)

    def __str__(self):
        return '(%d, %d)-Hamming-Code' % (self._N, self._n)

    ARRAY_LIKE = typing.Union[typing.List[float], np.ndarray, Vector]
