from __future__ import annotations

import typing
from itertools import combinations
from math import sqrt, acos, pi

import numpy as np


class Vector:
    elements: np.ndarray

    def __init__(self, *elements: float) -> None:
        self.elements = np.array(elements)

    def __add__(self, addend: ADDEND) -> Vector:
        if isinstance(addend, typing.List):
            return self.__add__(Vector(*addend))

        if isinstance(addend, np.ndarray):
            return self.__add__(Vector(*addend))

        if not isinstance(addend, Vector):
            raise TypeError('Argument must be of type ' + str(ADDEND))

        return Vector(*(self.elements + addend.elements))

    def __radd__(self, addend: ADDEND) -> Vector:
        return self.__add__(addend)

    def __sub__(self, addend: ADDEND) -> Vector:
        if not isinstance(addend, Vector):
            return self.__add__(np.array(addend) * (-1))

        return self.__add__(addend * (-1))

    def __rsub__(self, addend: ADDEND) -> Vector:
        if not isinstance(addend, Vector):
            return Vector(*np.array(addend)) + (self * (-1))

        return addend + (self * (-1))

    def __mul__(self, scalar: float) -> Vector:
        return Vector(*(self.elements * scalar))

    def __rmul__(self, scalar: float) -> Vector:
        return self.__mul__(scalar)

    def __getitem__(self, item: int) -> float:
        return self.elements[item].item()

    def __str__(self) -> str:
        return str(self.elements)

    def dot(self, other: ARRAY_LIKE) -> float:
        if isinstance(other, Vector):
            return self.dot(other.elements)

        return self.elements.dot(other)

    def norm(self) -> float:
        return sqrt(self.dot(self))

    def unit(self) -> Vector:
        return 1 / self.norm() * self

    def linearlyIndependent(self, other: ARRAY_LIKE) -> bool:
        return not self.linearlyDependent(other)

    def linearlyDependent(self, other: ARRAY_LIKE) -> bool:
        if not isinstance(other, Vector):
            return self.linearlyDependent(Vector(*np.array(other)))

        if self.isZeroVector() or other.isZeroVector():
            return True

        index = np.nonzero(self.elements)[0][0]

        if other.elements[index] == 0:
            return False

        alpha = other.elements[index] / self.elements[index]

        arr = []
        for i in range(len(self.elements)):
            arr.append(self.elements[i] * alpha == other.elements[i])

        return np.all(arr)

    def angle(self) -> float:
        alpha = acos(self.elements[0] / self.norm()) * 180 / pi
        return alpha if self.elements[1] > 0 else 360 - alpha

    def angleBetween(self, other: ARRAY_LIKE) -> float:
        otherVector = other if isinstance(other, Vector) else Vector(*other)
        return acos(self.dot(otherVector) * (1 / (self.norm() * otherVector.norm()))) * 180 / pi

    def angleBetweenRAD(self, other: ARRAY_LIKE) -> float:
        otherVector = other if isinstance(other, Vector) else Vector(*other)
        return acos(self.dot(otherVector) * (1 / (self.norm() * otherVector.norm())))

    def isZeroVector(self) -> bool:
        return all([e == 0 for e in self.elements])

    def homog(self) -> Vector:
        if len(self) == 0:
            return self

        return Vector(*np.append(self.elements, 1))

    def deHomog(self) -> Vector:
        length = len(self)
        lastCoordinate: float = (self[length - 1])
        return 1 / lastCoordinate * Vector(*np.delete(self.elements, length - 1))

    def __eq__(self, other):
        if isinstance(other, typing.List):
            return self == Vector(*other)

        if isinstance(other, np.ndarray):
            return self == Vector(*other)

        if len(self.elements) != len(other.elements):
            return False

        return np.allclose(self.elements, other.elements)

    def __len__(self):
        return len(self.elements)

    def __iter__(self):
        return iter(self.elements)

    @staticmethod
    def ogb(*vectors: Vector) -> bool:
        pairs = combinations(vectors, 2)

        # (i) Prüfen ob überhaupt eine
        # Basis gebildet werden kann
        if len(vectors[0]) != len(vectors):
            return False

        # (i) Prüfe ob es sich dabei um
        # Basisvektoren handelt (linear
        # unabhängig)
        if np.linalg.det((np.transpose(np.array(vectors)))) == 0:
            return False

        # (ii)
        for v, w in pairs:
            if v.dot(w) != 0:
                return False

        return True

    @staticmethod
    def onb(*vectors: Vector) -> bool:
        if not Vector.ogb(*vectors):
            return False

        for v in vectors:
            # (iii)
            if v.norm() != 1:
                return False

        return True


ADDEND = typing.Union[typing.List[float], typing.List[int], np.ndarray, Vector]
ARRAY_LIKE = typing.Union[typing.List[float], np.ndarray, Vector]
