# import os
# import imageio
import matplotlib as mpl  # type: ignore
import matplotlib.pyplot as plt
import numpy as np

from linalg.Matrix import Matrix
from linalg.plot.figures.cube import cubeEdges, cubePoints

PLOT_DPI = 200
mpl.rcParams['figure.dpi'] = PLOT_DPI


def projektionsmatrix(richtung, h):
    v1, v2, v3 = richtung
    return Matrix([1, 0, -v1 / v3, v1 * h / v3],
                  [0, 1, -v2 / v3, v2 * h / v3],
                  [0, 0, 0, h])


def xyEbene(range, h):
    x, y = np.meshgrid(
        np.linspace(*range, 10),
        np.linspace(*range, 10)
    )
    z = x * 0 + h
    return x, y, z


def drawLine(ax, p1, p2, color='gray', linestyle='dashed', linewidth=0.2, **kwargs):
    x1, y1, z1 = p1
    x2, y2, z2 = p2
    ax.plot([x1, x2], [z1, z2], [y1, y2], color=color, linestyle=linestyle, linewidth=linewidth, **kwargs)


def drawLines(ax, lines, **kwargs):
    for line in lines:
        drawLine(ax, *line)


def drawCube(ax, cubeEdges, colorPoints='green', colorEdges='darkgray'):
    for edge in cubeEdges:
        x1, y1, z1 = edge[0]
        x2, y2, z2 = edge[1]
        ax.plot([x1, x2], [z1, z2], [y1, y2], color=colorEdges)
        ax.scatter([x1, x2], [z1, z2], [y1, y2], color=colorPoints, s=2)


def drawXyPlane(ax, range, h, color='darkgray', alpha=0.2, **kwargs):
    x, y, z = xyEbene(range, h)
    ax.plot_surface(x, z, y, color=color, alpha=alpha, **kwargs)


def makePaneAndAxisTransparent(ax):
    # make the panes transparent
    ax.xaxis.set_pane_color((1.0, 1.0, 1.0, 0.0))
    ax.yaxis.set_pane_color((1.0, 1.0, 1.0, 0.0))
    ax.zaxis.set_pane_color((1.0, 1.0, 1.0, 0.0))
    # make the grid lines transparent
    ax.xaxis._axinfo["grid"]['color'] = (1, 1, 1, 0)
    ax.yaxis._axinfo["grid"]['color'] = (1, 1, 1, 0)
    ax.zaxis._axinfo["grid"]['color'] = (1, 1, 1, 0)


def coordinateSystem3d(range, axisOff=False):
    # create the figure
    fig = plt.figure()

    # add axes
    ax = fig.add_subplot(projection='3d')

    if axisOff:
        ax.set_axis_off()

    ax.set_xlim3d(range)
    ax.set_ylim3d(range)
    ax.set_zlim3d(range)
    ax.set_xlabel('X axis')
    ax.set_ylabel('Y axis')
    ax.set_zlabel('Z axis')
    return ax


# NOTE
# In order to use this code, you need to import
# `os` and `imageio`
#
# def gifPlot(plt):
#     filenames = []
#
#     for i in range(0, 360, 1):
#         ax.view_init(elev=0, azim=i)
#         filename = f'{i}.png'
#         filenames.append(filename)
#         plt.savefig("%d.png" % i, bbox_inches='tight')
#         print("Saved %d" % i)
#
#     with imageio.get_writer('pp3d.gif', mode='I') as writer:
#         for filename in filenames:
#             image = imageio.imread(filename)
#             writer.append_data(image)
#             print("Gif..." + filename)
#
#     # Remove files
#     for filename in set(filenames):
#         os.remove(filename)


def plotParallelprojektion(h=20, richtung=None):
    if richtung is None:
        richtung = [1, 2, 3]
    range3d = [-10, 25]
    planeRange3d = [-15, 30]
    M = projektionsmatrix(richtung, h)
    cubeEdgesTransformed = ([[M * point.homog() for point in edge] for edge in cubeEdges])
    projectionLines = zip(cubePoints, map(lambda p: (M * p.homog()), cubePoints))
    ax = coordinateSystem3d(range3d, axisOff=True)
    drawXyPlane(ax, planeRange3d, h)
    drawCube(ax, cubeEdges)
    drawCube(ax, cubeEdgesTransformed, colorPoints='black')
    drawLines(ax, projectionLines)
    ax.view_init(elev=0, azim=-15)
    ax.text2D(0.1, 0.05, "Parameter: $h=%d, richtung=%s^T$\nAuthor: Arbër Osmani" % (h, richtung), size=6, transform=ax.transAxes)
    # gifPlot(plt)
    #plt.savefig("plot.svg", bbox_inches='tight')
    plt.show()
