from __future__ import annotations
from linalg.Matrix import Matrix
from linalg.Vector import Vector
from math import sin, cos

import typing
import numpy as np


def hScale(factor: float) -> Matrix:
    return scale(factor, 1)


def vScale(factor: float) -> Matrix:
    return scale(1, factor)


def scale(h: float, v: float) -> Matrix:
    return Matrix([h, 0], [0, v])


def zoom(factor: float) -> Matrix:
    return scale(factor, factor)


def hShear(factor: float) -> Matrix:
    return shear(factor, 0)


def vShear(factor: float) -> Matrix:
    return shear(0, factor)


def shear(h: float, v: float) -> Matrix:
    return Matrix([1, h], [v, 1])


def rotation(alpha: float) -> Matrix:
    return Matrix([cos(alpha), -sin(alpha)],
                  [sin(alpha), cos(alpha)])


def reflection(alpha: float) -> Matrix:
    return Matrix([cos(2 * alpha), sin(2 * alpha)],
                  [sin(2 * alpha), -cos(2 * alpha)])


def identity(n: int) -> Matrix:
    return Matrix(*np.identity(n))


def translation(v: ARRAY_LIKE) -> Matrix:
    idm: Matrix = identity(len(v))
    column: np.ndarray = v.elements if isinstance(v, Vector) else np.array(v)
    row = np.append(np.zeros(len(v)), 1)

    return idm.appendCol(column).appendRow(row)


def scale3d(x: float, y: float, z: float) -> Matrix:
    return Matrix([x, 0, 0], [0, y, 0], [0, 0, z])


def xScale3d(x: float) -> Matrix:
    return scale3d(x, 1, 1)


def yScale3d(y: float) -> Matrix:
    return scale3d(1, y, 1)


def zScale3d(z: float) -> Matrix:
    return scale3d(1, z, 1)


def xRotation3d(alpha: float) -> Matrix:
    return Matrix([1, 0, 0],
                  [0, cos(alpha), -sin(alpha)],
                  [0, sin(alpha), cos(alpha)])


def yRotation3d(alpha: float) -> Matrix:
    return Matrix([cos(alpha), 0, sin(alpha)],
                  [0, 1, 0],
                  [-sin(alpha), 0, cos(alpha)])


def zRotation3d(alpha: float) -> Matrix:
    return Matrix([cos(alpha), -sin(alpha), 0],
                  [sin(alpha), cos(alpha), 0],
                  [0, 0, 1])


def xyReflection3d() -> Matrix:
    return Matrix([1, 0, 0], [0, 1, 0], [0, 0, -1])


def yzReflection3d() -> Matrix:
    return Matrix([-1, 0, 0], [0, 1, 0], [0, 0, 1])


def xzReflection3d() -> Matrix:
    return Matrix([1, 0, 0], [0, -1, 0], [0, 0, 1])


def homog(map: Matrix) -> Matrix:
    return map.homog()


ARRAY_LIKE = typing.Union[typing.List[float], np.ndarray, Vector]
